<?php

namespace Modules\TpsTransfer\Http\Controllers;

use App\Brands;
use App\BusinessLocation;
use App\Category;
use App\SellingPriceGroup;
use App\TaxRate;
use App\Transaction;
use App\Unit;
use App\User;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Modules\TpsTransfer\Entities\NstPrepareStock;
use Modules\TpsTransfer\Entities\NstShipping;
use Modules\TpsTransfer\Entities\NstUserLocation;
use Modules\TpsTransfer\Entities\NstSetting;
use Modules\TpsTransfer\Entities\Services\LocationsService;
use Modules\TpsTransfer\Entities\Services\TransactionService;
use Spatie\Activitylog\Models\Activity;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Yajra\DataTables\DataTables;

class TpsController extends Controller
{
    protected $transactionService;
    protected $moduleUtil;
    protected $locationsService;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct(TransactionService $transactionService, ModuleUtil $moduleUtil, LocationsService $locationsService)
    {
        $this->moduleUtil = $moduleUtil;
        $this->transactionService = $transactionService;
        $this->locationsService = $locationsService;
    }

    /**
     * Show all products in New stock transfer module.
     */
    public function index()
    {
        if (!auth()->user()->can('tpsTransfer.access_index_stock_transfers')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = request()->session()->get('user.business_id');
        $is_woocommerce = $this->moduleUtil->isModuleInstalled('Woocommerce');

        if (request()->ajax()) {
            //Filter by location
            $from_location_id = request()->get('from_location_id', null);
            $to_location_id = request()->get('to_location_id', null);
            $shipping_id = request()->get('shipping_id', null);

            $query = NstPrepareStock::query()
                ->where('nst_prepare_products.business_id', $businessId)
                ->leftJoin('users','users.id','=','nst_prepare_products.created_by')
                ->leftJoin('business_locations as from_business_locations','from_business_locations.id','=','nst_prepare_products.from_location_id')
                ->leftJoin('products','products.id','=','nst_prepare_products.product_id')
                ->leftJoin('categories as c1', 'products.category_id', '=', 'c1.id')
                ->leftJoin('categories as c2', 'products.sub_category_id', '=', 'c2.id')
                ->leftJoin('brands', 'products.brand_id', '=', 'brands.id')
                ->join('units', 'products.unit_id', '=', 'units.id')
                ->leftJoin('tax_rates', 'products.tax', '=', 'tax_rates.id')
                ->leftJoin('variations','variations.id','=','nst_prepare_products.variation_id')
                ->join('product_variations as pv', 'pv.id', '=', 'variations.product_variation_id')
                ->leftJoin('purchase_lines','purchase_lines.id','=','nst_prepare_products.purchase_lines_id')
                ->leftJoin('business_locations as to_business_locations','to_business_locations.id','=','nst_prepare_products.to_location_id')
                ->whereNull('nst_prepare_products.deleted_at')

                ->leftJoin('variation_location_details as to_vld',function($join) {
                    $join->on('variations.id','=','to_vld.variation_id')
                        ->whereRaw('to_vld.location_id = to_business_locations.id');
                })
                ->leftJoin('variation_location_details as from_vld',function($join) {
                    $join->on('variations.id','=','from_vld.variation_id')
                        ->whereRaw('from_vld.location_id = from_business_locations.id');
                });

            if (!empty($shipping_id) && $shipping_id != 'all') {
                $query->where('nst_prepare_products.nst_shipping_id', '=', $shipping_id);
            }

            if (!empty($from_location_id) && $from_location_id != 'none') {
                if ($from_location_id == 'all') {
                    $fromBusinessLocations = $this->locationsService->accessLocationsForUserforDropdown($businessId, 'from', false, true, 'index');

                    $query->whereIn('nst_prepare_products.from_location_id', array_keys($fromBusinessLocations));
                }else{
                    $query->where('nst_prepare_products.from_location_id', '=', $from_location_id);
                }
            }

            if (!empty($to_location_id) && $to_location_id != 'none') {
                if ($to_location_id == 'all') {
                    $toBusinessLocations = $this->locationsService->accessLocationsForUserforDropdown($businessId, 'to', false, true, 'index');

                    $query->whereIn('nst_prepare_products.to_location_id', array_keys($toBusinessLocations));
                }else{
                    $query->where('nst_prepare_products.to_location_id', '=', $to_location_id);
                }
            }

            $nstStatus = request()->get('nst_status', null);
            if ($nstStatus && $nstStatus != 'all') {
                $query->where('nst_prepare_products.status', $nstStatus);
            }

            $query = $query->select(
                'nst_prepare_products.id',
                'nst_prepare_products.transaction_date',
                'nst_prepare_products.ref_no',
                'nst_prepare_products.is_shipping_calculated',
                'users.id as user_id',
                'users.surname as user_surname',
                'from_location_id',
                'from_business_locations.name as from_location_name',
                'products.id as product_id',
                'products.name as product_name',
                'products.sku',
                'products.type',
                'c1.name as category',
                'c2.name as sub_category',
                'units.actual_name as unit',
                'brands.name as brand',
                'tax_rates.name as tax',
                'variations.id as variation_id',
                'variations.name as variation_name',
                'pv.name as product_variation_name',
                'purchase_lines.lot_number',
                'purchase_lines.purchase_price',
                'purchase_lines.purchase_price_inc_tax',
                'nst_prepare_products.transfer_quantity',
                'to_location_id',
                'to_business_locations.name as to_location_name',
                'nst_prepare_products.is_important',
                'nst_prepare_products.note',
                'nst_prepare_products.value_per_unit',
                'nst_prepare_products.status',
                DB::raw('to_vld.qty_available as to_qty_available'),
                DB::raw('from_vld.qty_available as from_qty_available')
            );
            if (!request()->get('order')) {
                $query
                    ->orderBy('products.id', 'ASC')
                    ->orderBy('to_qty_available', 'ASC')
                    ->orderBy('nst_prepare_products.is_important', 'DESC')
                    ->orderByRaw('ISNULL(nst_prepare_products.note), nst_prepare_products.note ASC')
                ;
            };

            $type = request()->get('type', null);
            if (!empty($type)) {
                $query->where('products.type', $type);
            }

            $category_id = request()->get('category_id', null);
            if (!empty($category_id)) {
                $query->where('products.category_id', $category_id);
            }

            $categoryId = request()->get('category_id', null);
            $subCategoryId = request()->get('sub_category_id', null);

            if ($subCategoryId && $subCategoryId != 'all') {
                if ($subCategoryId) {
                    $query->where('products.sub_category_id', $subCategoryId);
                }
                if ($categoryId) {
                    $query->where('products.category_id', $categoryId);
                }
            }

            $brand_id = request()->get('brand_id', null);
            if (!empty($brand_id)) {
                $query->where('products.brand_id', $brand_id);
            }

            return \Yajra\DataTables\Facades\DataTables::of($query)
                ->addColumn('data-href', function ($row){
                    return action('\\App\Http\Controllers\ProductController@view', [$row->product_id]);
                })
                ->addColumn('row_checkbox', function ($row){
                    return "<input type='checkbox' class='product-row-input' value='1' data-id='{$row->id}' data-name='accepted'> ";
                })
                ->addColumn('value_per_unit', function ($row){
                    return round($row->value_per_unit, 2);
                })
                ->editColumn('from_location_name', function ($row){
                    return $row->from_location_name .' ('.intval($row->from_qty_available).')';
                })
                ->editColumn('to_location_name', function ($row){
                    return $row->to_location_name .' ('.intval($row->to_qty_available).')';
                })
                ->addColumn('actions', function ($row){
                    $class = is_null($row->note) ? '' : 'btn-success';
                    $html = "
                <div class='btn-group'>
                    <div class='dropdown actions-list'>
                        <button class='btn btn-primary dropdown-toggle' type='button' data-toggle='dropdown' id='actions_btn_{$row->id}'>".__('messages.actions')."</button>
                        <div class='dropdown-menu dropdown-menu-right'>
                        <li><a target='_blank' href='".action('\Modules\TpsTransfer\Http\Controllers\TpsLabelsController@show', ['product_id' => $row->product_id, 'qty' => intval($row->transfer_quantity)]) ."' data-toggle='tooltip' title='" . __('lang_v1.label_help') . "'><i class='fa fa-barcode'></i> " . __('barcode.labels') . "</a></li>
                        <li><a data-toggle='modal' data-target='#add_notes' id='note_{$row->id}' data-id='{$row->id}' class='btn dropdown-item add_notes {$class}' type='button'>".__('tpsTransfer::lang.note')."</a></li>
                        </div>
                    </div>
                </div>";
                    return $html;
                })
                ->editColumn('category', '{{$category}} @if(!empty($sub_category))<br/> -- {{$sub_category}}@endif')
                ->editColumn('product', function ($row) {
                    $name = $row->product;
                    if ($row->variation_name && $row->variation_name != 'DUMMY') {
                        $name .= ' (' . $row->variation_name . ')';
                    }
                    return $name;
                })
                ->editColumn('image', function ($row) {
                    return '<div style="display: flex;"><img src="' . $row->image_url . '" alt="Product image" class="product-thumbnail-small"></div>';
                })
                ->editColumn('purchase_price', function ($row) use ($businessId) {
                    $shipping = ($row->is_shipping_calculated == 1) ? " <i class='fa fa-shipping-fast'>" : null;

                    if($row->purchase_price){
                        return $row->purchase_price.$shipping;
                    }

                    //add lot select list
                    $avgPurchasePrice = 0;
                    $count = 0;

                    $lotNumbers = $this->transactionService->getLotDataFromVariation($row->variation_id, $businessId, $row->from_location_id, true);

                    if($lotNumbers){
                        foreach ($lotNumbers as $lotNumber) {
                            $avgPurchasePrice += $lotNumber->purchase_price * $lotNumber->qty_available;
                            $count += $lotNumber->qty_available;
                        }
                        if($count){
                            $avgPurchasePrice = $avgPurchasePrice / $count;
                        }
                    }
                    return round($avgPurchasePrice, 2).$shipping;
                })
                ->editColumn('type', '@lang("lang_v1." . $type)')
                ->editColumn('transfer_quantity', function ($row) {
                    return intval($row->transfer_quantity);
                })
                ->filterColumn('products.sku', function ($query, $keyword) {
                    $query->whereHas('product.variations', function($q) use($keyword){
                        $q->where('sub_sku', 'like', "%{$keyword}%");
                    })
                        ->orWhere('products.sku', 'like', "%{$keyword}%");
                })
                ->editColumn('status', function($row) {
                    switch ($row->status) {
                        case "receive":
                            $color = 'blue';
                            $status = ucfirst($row->status);
                            break;
                        case "complete":
                            $color = 'green';
                            $status = ucfirst($row->status);
                            break;
                        default:
                            $color = 'gray';
                            $status = ucfirst($row->status);
                    }

                    return "<span class='label bg-{$color}'>{$status}</span>";
                })
                ->setRowAttr([
                    'data-href' => function ($row) {
                        return  action('\Modules\TpsTransfer\Http\Controllers\TpsController@showInfoModal', [$row->id]) ;
                    }])
                ->rawColumns(['actions', 'row_checkbox', 'product', 'selling_price', 'purchase_price', 'category', 'status'])
                ->make(true);
        }


        $categories = Category::forDropdown($businessId, 'product');
        $brands = Brands::forDropdown($businessId);
        $units = Unit::forDropdown($businessId);
        $tax_dropdown = TaxRate::forBusinessDropdown($businessId, false);
        $taxes = $tax_dropdown['tax_rates'];

        $fromBusinessLocations = $this->locationsService->accessLocationsForUserforDropdown($businessId, 'from', false, true, 'index', null, true);
        $toBusinessLocations = $this->locationsService->accessLocationsForUserforDropdown($businessId, 'to', false, true, 'index');


        return view('tpsTransfer::index')->with(compact(
            'categories',
            'brands',
            'units',
            'taxes',
            'fromBusinessLocations',
            'toBusinessLocations',
            'is_woocommerce',
        ));
    }


    /**
     * Show all transactions done by New stock transfer module.
     */
    public function transactions(Request $request)
    {
        if (!auth()->user()->can('tpsTransfer.access_transactions_stock_transfers')) {
            abort(403, 'Unauthorized action.');
        }

        $businessId = request()->session()->get('user.business_id');
        $statuses = ['completed' => __('restaurant.completed')];
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $edit_days = request()->session()->get('business.transaction_edit_days');
            $all_transactions = $request->all_transactions ?: null;

            $stock_transfers = Transaction::join(
                'business_locations AS l1',
                'transactions.location_id',
                '=',
                'l1.id'
            )
                ->join('transactions as t2', 't2.transfer_parent_id', '=', 'transactions.id')
                ->join(
                    'business_locations AS l2',
                    't2.location_id',
                    '=',
                    'l2.id'
                )
                ->groupby('transactions.id')
                ->where('transactions.business_id', $business_id)
                ->where('transactions.type', 'sell_transfer');

            if(empty($request->from_location_id) || empty($request->to_location_id)){
	            $stock_transfers->whereRaw('1 = 2');
            }
            if (!$all_transactions) {
                $stock_transfers->join('nst_prepare_products as nst', 'nst.transaction_id', '=', 'transactions.id');

                if($request->from_location_id && $request->from_location_id !== 'all'){
                    $stock_transfers->where('nst.from_location_id', $request->from_location_id);
                }
                if($request->to_location_id && $request->to_location_id !== 'all'){
                    $stock_transfers->where('nst.to_location_id', $request->to_location_id);
                }
                if($request->shipping_id && $request->shipping_id !== 'all'){
                    $stock_transfers->where('nst.nst_shipping_id', $request->shipping_id);
                }
            }

            $stock_transfers->select(
                'transactions.id',
                'transactions.transaction_date',
                'transactions.ref_no',
                'l1.name as location_from',
                'l2.name as location_to',
                'transactions.final_total',
                'transactions.shipping_charges',
                'transactions.additional_notes',
                'transactions.id as DT_RowId',
                'transactions.status'
            );
            if (!request()->get('order')) {
                $stock_transfers->orderBy('transactions.id', 'DESC');
            }
            return DataTables::of($stock_transfers)
                ->addColumn('action', function ($row) use ($edit_days) {
                    $html = '<button type="button" title="' . __("stock_adjustment.view_details") . '" class="btn btn-primary btn-xs btn-modal" data-container=".view_modal" data-href="' . action('\Modules\TpsTransfer\Http\Controllers\TpsController@showStockTransferModal', [$row->id]) . '"><i class="fa fa-eye" aria-hidden="true"></i> ' . __('messages.view') . '</button>';
                    $html .= ' <a href="#" class="print-invoice btn btn-info btn-xs" data-href="' . action('\Modules\TpsTransfer\Http\Controllers\StockTransferController@printInvoice', [$row->id]) . '"><i class="fa fa-print" aria-hidden="true"></i> '. __("messages.print") .'</a>';

                    return $html;
                })
                ->editColumn(
                    'final_total',
                    '<span class="display_currency" data-currency_symbol="true">{{$final_total}}</span>'
                )
                ->editColumn(
                    'shipping_charges',
                    '<span class="display_currency" data-currency_symbol="true">{{$shipping_charges}}</span>'
                )
                ->editColumn('status', function($row) {
                    return '<span class="label bg-green">completed</span>';
                })
                ->editColumn('transaction_date', '{{@format_datetime($transaction_date)}}')
                ->rawColumns(['final_total', 'action', 'shipping_charges', 'status'])
                ->setRowAttr([
                    'data-href' => function ($row) {
                        return  action('\Modules\TpsTransfer\Http\Controllers\TpsController@showStockTransferModal', [$row->id]);
                    }])
                ->make(true);
        }

        $fromBusinessLocations = $this->locationsService->accessLocationsForUserforDropdown($businessId, 'from', false, true, 'complete', null, true, true);

        return view('tpsTransfer::transactions')->with(compact(
            'statuses',
            'fromBusinessLocations'
        ));
    }

    /**
     * Show product modal info.
     * @return Response
     */
    public function showInfoModal($id)
    {
        $businessId = request()->session()->get('user.business_id');

        $nstProduct = NstPrepareStock::where('business_id', $businessId)
            ->with(['product', 'product.brand', 'product.unit', 'product.category', 'product.sub_category', 'product.product_tax', 'product.variations', 'product.purchase_lines', 'product.media', 'shipping', 'to_location', 'from_location'])
            ->findOrFail($id);

        if($nstProduct->purchase_price){
            $purchasePrice = $nstProduct->purchase_price;
        }else{
            $count = 0;
            $avg = 0;

            $lotNumbers = $this->transactionService->getLotDataFromVariation($nstProduct->variation_id, $businessId, $nstProduct->from_location_id, true);

            if($lotNumbers){
                foreach ($lotNumbers as $lotNumber) {
                    $avg += $lotNumber->purchase_price * $lotNumber->qty_available;
                    $count += $lotNumber->qty_available;
                }
            }
            $purchasePrice = $count ? round($avg / $count, 2) : $avg;
        }

        $transferQuantity = $nstProduct->transfer_quantity;

        $activities = Activity::forSubject($nstProduct)
            ->with(['causer', 'subject'])
            ->latest()
            ->get();

        return view('tpsTransfer::nst-info-modal')->with(compact(
            'nstProduct', 'purchasePrice', 'transferQuantity', 'activities'
        ));
    }

    /**
     * Show completed stock transfer info
     * @return Response
     */
    public function showStockTransferModal($id)
    {
        if (!auth()->user()->can('purchase.view')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $sell_transfer = Transaction::where('business_id', $business_id)
            ->where('id', $id)
            ->where('type', 'sell_transfer')
            ->with(
                'contact',
                'sell_lines',
                'sell_lines.product',
                'sell_lines.variations',
                'sell_lines.variations.product_variation',
                'sell_lines.lot_details',
                'sell_lines.sub_unit',
                'location',
                'sell_lines.product.unit'
            )
            ->first();

        foreach ($sell_transfer->sell_lines as $key => $value) {
            if (!empty($value->sub_unit_id)) {
                $formated_sell_line = $this->transactionUtil->recalculateSellLineTotals($business_id, $value);

                $sell_transfer->sell_lines[$key] = $formated_sell_line;
            }
        }

        $purchase_transfer = Transaction::where('business_id', $business_id)
            ->where('transfer_parent_id', $sell_transfer->id)
            ->where('type', 'purchase_transfer')
            ->first();

        $location_details = ['sell' => $sell_transfer->location, 'purchase' => $purchase_transfer->location];

        $lot_n_exp_enabled = false;
        if (request()->session()->get('business.enable_lot_number') == 1 || request()->session()->get('business.enable_product_expiry') == 1) {
            $lot_n_exp_enabled = true;
        }

        $statuses['final'] = __('restaurant.completed');

        $activities = Activity::forSubject($sell_transfer)
            ->with(['causer', 'subject'])
            ->latest()
            ->get();

        return view('tpsTransfer::cloned_blades.transfer_info')
            ->with(compact('sell_transfer', 'location_details', 'lot_n_exp_enabled', 'statuses', 'activities'));
    }

    public function setting()
    {
        if (!auth()->user()->can('roles.view') && !auth()->user()->can('tpsTransfer.access_settings_stock_transfer')) {
            abort(403, 'Unauthorized action.');
        }
        $businessId = request()->session()->get('user.business_id');
        $setting = NstSetting::where('business_id', $businessId)->first();
        $setting = $setting ? $setting->value : [];
        $shippingMethods = [
            'as_charge' => 'keep it as shipping charge',
            'different_ratio' => 'divided by different ratio',
            'equal_ratio' => 'divided equal ratio',
        ];
        return view('tpsTransfer::nst-setting')->with(compact('setting', 'shippingMethods'));
    }

    public function storeSetting(Request $request)
    {
        $businessId = request()->session()->get('user.business_id');

        try {
            $value = [
                'sellingPeriod' => $request->sellingPeriod,
                'futurePeriod' => $request->futurePeriod,
                'shippingMethod' => $request->shippingMethod,
                'disableDefaultStockTransfer' => $request->disableDefaultStockTransfer,
            ];

            NstSetting::updateOrCreate(
                ['business_id' => $businessId],
                ['value' => $value]
            );

            $output = ['success' => 1,
                'msg' => 'Setting updated successfully.'
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency("File:" . $e->getFile(). "Line:" . $e->getLine(). "Message:" . $e->getMessage());

            $output = ['success' => 0,
                'msg' => $e->getMessage()
            ];
            return back()->with('status', $output);
        }
        return back()->with('status', $output);
    }

    //permission assign to roles
    public function indexUsers()
    {
        if (!auth()->user()->can('roles.view') && !auth()->user()->can('tpsTransfer.access_settings_stock_transfer')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $users = User::where('business_id', $business_id)
            ->user()
            ->select(['id', 'username',
                DB::raw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) as full_name"), 'email', 'allow_login']);

        return \Yajra\DataTables\Facades\DataTables::of($users)
            ->editColumn('username', '{{$username}} @if(empty($allow_login)) <span class="label bg-gray">@lang("lang_v1.login_not_allowed")</span>@endif')
            ->addColumn('role', function ($row) {
                $role_name = $this->moduleUtil->getUserRoleName($row->id);
                return $role_name;
            })
            ->addColumn('action', function ($row) {
                return '<a href="' . action('\Modules\TpsTransfer\Http\Controllers\TpsController@editUserPermission', [$row->id]) . '" class="btn btn-xs btn-primary"><i class="glyphicon glyphicon-edit"></i> ' . __("messages.edit") . '</a>';
            })
            ->filterColumn('full_name', function ($query, $keyword) {
                $query->whereRaw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) like ?", ["%{$keyword}%"]);
            })
            ->removeColumn('id')
            ->rawColumns(['action', 'username'])
            ->make(true);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function editUserPermission($id)
    {
        $business_id = request()->session()->get('user.business_id');

        if (!auth()->user()->can('roles.update')) {
            abort(403, 'Unauthorized action.');
        }
        $user = User::where('business_id', $business_id)
            ->with(['permissions'])
            ->find($id);

        $types = [
            'index' => 'Index page',
            'add' => 'Add page',
            'prepare' => 'Prepare page',
            'receive' => 'Receive page',
        ];

        $user_permissions = [];
        foreach ($user->permissions as $user_perm) {
            $user_permissions[] = $user_perm->name;
        }

        $class = 'Modules\TpsTransfer\Http\Controllers\DataController';
        $module_permissions['TpsTransfer'] = call_user_func([new $class(), 'user_permissions']);
        $locations = BusinessLocation::where('business_id', $business_id)->pluck('name', 'id');

        $userLocations = NstUserLocation::where(
            ['business_id' => $business_id, 'user_id' => $id]
        )->get();

        $userPermissions = $user->permissions()->where('name', 'like', 'tpsTransfer.location_access_%')->pluck('name');

        return view('tpsTransfer::user-permission-edit')
            ->with(compact('user', 'user_permissions', 'module_permissions', 'types', 'locations', 'userLocations', 'userPermissions'));
    }

    /**
     * Store or update access locations for specific user
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function storeUserLocation($id)
    {
        if (!auth()->user()->can('roles.update')) {
            abort(403, 'Unauthorized action.');
        }
        $business_id = request()->session()->get('user.business_id');

        $user = User::where('business_id', $business_id)
            ->find($id);

        $oldPermissions = $user->permissions()->where('name', 'like', 'tpsTransfer.location_access_%')->get();

        $user->revokePermissionTo($oldPermissions);

        $permissions = request()->data;

        $this->__createPermissionIfNotExists($permissions);


        if (!empty($permissions)) {
            $user->givePermissionTo($permissions);
        }

        $locations = request()->locations ? request()->locations : [];
        NstUserLocation::where(
            ['business_id' => $business_id, 'user_id' => $user->id]
        )->delete();
        foreach($locations as $index => $value){
            foreach ($value as $location_id){
                NstUserLocation::firstOrCreate(
                    ['business_id' => $business_id, 'user_id' => $user->id, 'type' => $index, 'location_id' => $location_id]
                );
            }
        }

        $output = ['success' => 1,
            'msg' => __("user.user_updated")
        ];
        return back()->with('status', $output);
    }

    /**
     * Creates new permission if doesn't exist
     *
     * @param  array  $permissions
     * @return void
     */
    private function __createPermissionIfNotExists($permissions)
    {
        $exising_permissions = Permission::whereIn('name', $permissions)
            ->pluck('name')
            ->toArray();

        $non_existing_permissions = array_diff($permissions, $exising_permissions);

        if (!empty($non_existing_permissions)) {
            foreach ($non_existing_permissions as $new_permission) {
                $time_stamp = \Carbon::now()->toDateTimeString();
                Permission::create([
                    'name' => $new_permission,
                    'guard_name' => 'web'
                ]);
            }
        }
    }

    /**
     * get activity by arguments
     */
    public function getActivityByArguments(Request $request)
    {
        $nstPrepareStock = NstPrepareStock::where('product_id', $request->product_id)->where('transaction_id', $request->transaction_id)->where('variation_id', $request->variation_id)
            ->first();

        return $this->showInfoModal($nstPrepareStock->id);
    }

    /**
     * get to location by from location id
     */
    public function getToLocationByFromLocationId(Request $request)
    {
        $businessId = request()->session()->get('user.business_id');
        $to_locations = $this->locationsService->accessLocationsForUserforDropdown($businessId, 'to', false, true, $request->status == 'all' ? null : $request->status, $request->from_location_id, $request->preAppendAll);

        $html = '';
        if (!empty($to_locations)) {
            foreach ($to_locations as $key => $index) {
                $html .= "<option value='{$key}'> {$index} </option>";
            }
        }
        echo $html;
        exit;
    }

    /**
     * get shipping ID by from and to locations ID
     */
    public function getShippingIdsWithLocations(Request $request)
    {
        $businessId = request()->session()->get('user.business_id');
        $shippingIds = NstShipping::query()
            ->where('nst_shipping.business_id', $businessId)
            ->whereHas('nstPrepareStock', function ($query) use ($request) {
                if($request->status){
                    $query->where('status', $request->status);
                }
                if($request->from_location_id != 'all'){
                    $query->where('from_location_id', $request->from_location_id);
                }

                if($request->to_location_id != 'all'){
                    $query->where('to_location_id', $request->to_location_id);
                }
            })
            ->select("nst_shipping.id", "nst_shipping.ref_num")
            ->groupBy("nst_shipping.id")
            ->pluck('nst_shipping.ref_num', 'nst_shipping.id');

        $html = '';

        if($request->preAppendAll){
            $html .= "<option value='all'> ".__('lang_v1.all')." </option>";
        }

        if($request->preAppendNone){
            $html .= "<option value='none'> ".__('tpsTransfer::lang.without_shipping')." </option>";
        }
        foreach ($shippingIds as $key => $value) {
            $html .= "<option value='{$key}'> {$value} </option>";
        }
        echo $html;
        exit;
    }


}
